/*
 * Copyright (c) 2018 Guillem96
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdarg.h>
#include <string.h>

#include "gfx/di.h"
#include "gfx/gfx.h"

#include "mem/heap.h"
#include "mem/sdram_lp0_param_t210.h"

#include "soc/hw_init.h"
#include "soc/t210.h"
#include "soc/pmc.h"

#include "core/launcher.h"

#include "utils/util.h"
#include "utils/fs_utils.h"
#include "utils/btn.h"

#define CENTERED_TEXT_START(n) (((1280 - n * CHAR_WIDTH * g_gfx_con.scale) / 2))

extern void pivot_stack(u32 stack_top);
extern u8 get_payload_num(void);
void sdram_lp0_save_params(const void *params);

bool g_render_embedded_splash = true;

static inline void setup_gfx()
{
    u32 *fb = display_init_framebuffer();
    gfx_init_ctxt(&g_gfx_ctxt, fb, 1280, 720, 720);
    gfx_clear_color(&g_gfx_ctxt, 0xFF000000);
    gfx_con_init(&g_gfx_con, &g_gfx_ctxt);
    g_gfx_con.scale = 1;
}

void display_logo_with_message(int count, ...)
{
    /* vardiac arg list for multiple strings. */
    va_list strs;
    va_start(strs, count);

    /* save the scale factor so we can restore it later. */
    u8 scale_old = g_gfx_con.scale;
    
    /* Render the embedded splash, if needed */
    if(g_render_embedded_splash)
    {
        /* set scale and position for logo */
        g_gfx_con.scale = 3;
        g_gfx_con.x = 370;
        g_gfx_con.y = 98;

        /* print logo.                                             */
        /* Logo stored as chars, which are included with the font, */
        /* the chars are encoded outside of normal ascii range.    */
        /* The logo is comprised of 4 rows and 12 columns.         */
        for(char i = 0; i < 4; i++)
        {
            for(char j = 0; j < 12; j++)
            {
                gfx_putc(&g_gfx_con, (i * 12) + j + 0x7F);
            }
            gfx_putc(&g_gfx_con, '\n');
            g_gfx_con.x = 370;
        }
    }

    /* set scale and y position for text. */
    g_gfx_con.scale = 1;
    g_gfx_con.y = 593;

    /* Allocate space for pointers to each string,                   */
    /* along with var to hold total number of chars to print         */
    /* and starting index to keep track of already processed strings */
    char **str_ptrs = malloc(count * sizeof(char *));
    int start = 0;
    int total_len = 0;
    for(int i = 0; i <= count; i++)
    {
        /* Get next string from va list */
        char *str = va_arg(strs, char *);
        str_ptrs[i] = str; 

        /* Handle newlines separately, so text centering isn't thrown off, */
        /* and print characters that haven't been printed yet.             */
        if(str[0] == '\n' || i == count)
        {
            /* Set text position. Center text horizontally. */
            g_gfx_con.x = CENTERED_TEXT_START(total_len);

            /* Print each string, starting at calculated x position. */
            for(int j = start; j < i; j++)
                gfx_puts(&g_gfx_con, str_ptrs[j]);

            /* Next string to worry about is after current string.             */
            /* Reset the number of characters to print to zero, for next line. */
            start = i + 1;
            total_len = 0;

            gfx_putc(&g_gfx_con, '\n');
            continue;
        }

        total_len += strlen(str);
    }
    /* Free list of pointers. */
    free(str_ptrs);
    str_ptrs = 0;

    /* Restore the scale factor. */
    g_gfx_con.scale = scale_old;
}

void ipl_main()
{
    config_hw();

    /* Init the stack and the heap */
    pivot_stack(0x90010000);
    heap_init(0x90020000);


    if(PMC(APBDEV_PMC_SCRATCH116) == 0xDEADBEEF)
    {

        PMC(APBDEV_PMC_SCRATCH116) = 0;   

        /* Init display and gfx module */
        display_init();
        setup_gfx();
        display_backlight_pwm_init();
        display_backlight_brightness(100, 1000);

        g_gfx_con.fgcol = 0xFF00FF00;
        display_logo_with_message(5, "dMMC update mode", "\n", "Connect console to PC to change settings or upgrade.", "\n", "Press volume to reboot.");
    }
    else
    {
        /* Arbwrite 1 */
        PMC(APBDEV_PMC_SCRATCH45) = 0x2E38DFFF; // ipatch: 0x105C70 = 0xDFFF (SVC 0xFF)
        PMC(APBDEV_PMC_SCRATCH46) = 0x6001DC28; // overwrite ipatch slot 9

        /* Arbwrite 2 */
        PMC(APBDEV_PMC_SCRATCH33) = 0x40010000; // payload location
        PMC(APBDEV_PMC_SCRATCH40) = 0x6000F208; // overwrite BPMP SVC

        PMC(APBDEV_PMC_SCRATCH116) = 0xDEADBEEF; // magic number for indicating we've already rebooted

        PMC(APBDEV_PMC_SCRATCH0) |= 1; // set bit 0 to signify warmboot on reboot
        PMC(APBDEV_PMC_CNTRL) = PMC_CNTRL_MAIN_RST; // reboot
        while(1);
    }

    btn_wait();
    PMC(APBDEV_PMC_SCRATCH0) |= 2; // set bit 1 to signify RCM mode
    PMC(APBDEV_PMC_CNTRL) = PMC_CNTRL_MAIN_RST; // reboot
    while(1);
}